<?php

use App\Models\Amenity;
use App\Models\Deal;
use App\Models\Document;
use App\Models\Lead;
use App\Models\Lease;
use App\Models\Property;
use App\Models\PropertyPhoto;
use App\Models\User;

it('can create a property with all fields', function () {
    $property = Property::factory()->create([
        'property_type' => 'residential',
        'status' => 'available',
        'bedrooms' => 3,
    ]);

    expect($property->property_type)->toBe('residential')
        ->and($property->status)->toBe('available')
        ->and($property->bedrooms)->toBe(3);
});

it('belongs to an agent', function () {
    $agent = User::factory()->create(['role' => 'agent']);
    $property = Property::factory()->create(['agent_id' => $agent->id]);

    expect($property->agent->id)->toBe($agent->id);
});

it('has photos', function () {
    $property = Property::factory()->create();
    PropertyPhoto::factory()->primary()->create(['property_id' => $property->id]);
    PropertyPhoto::factory()->count(3)->create(['property_id' => $property->id]);

    expect($property->photos)->toHaveCount(4)
        ->and($property->photos->where('is_primary', true))->toHaveCount(1);
});

it('has amenities via pivot', function () {
    $property = Property::factory()->create();
    $amenities = Amenity::factory()->count(3)->create();
    $property->amenities()->attach($amenities->pluck('id'));

    expect($property->amenities)->toHaveCount(3);
});

it('has deals', function () {
    $property = Property::factory()->create();
    Deal::factory()->count(2)->create(['property_id' => $property->id]);

    expect($property->deals)->toHaveCount(2);
});

it('has leases', function () {
    $property = Property::factory()->create();
    Lease::factory()->count(2)->create(['property_id' => $property->id]);

    expect($property->leases)->toHaveCount(2);
});

it('has leads', function () {
    $property = Property::factory()->create();
    Lead::factory()->count(3)->create(['property_id' => $property->id]);

    expect($property->leads)->toHaveCount(3);
});

it('has polymorphic documents', function () {
    $property = Property::factory()->create();
    Document::factory()->create([
        'documentable_type' => Property::class,
        'documentable_id' => $property->id,
    ]);

    expect($property->documents)->toHaveCount(1);
});
