<?php

namespace Database\Seeders;

use App\Models\Account;
use App\Models\AccountTransaction;
use App\Models\Amenity;
use App\Models\Client;
use App\Models\ClientInteraction;
use App\Models\Deal;
use App\Models\DealStageHistory;
use App\Models\Document;
use App\Models\Lead;
use App\Models\Lease;
use App\Models\LeasePayment;
use App\Models\Property;
use App\Models\PropertyPhoto;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Database\Seeder;

class DatabaseSeeder extends Seeder
{
    /**
     * @var list<string>
     */
    private array $dealStages = ['inquiry', 'viewing', 'offer', 'negotiation', 'agreement', 'closing', 'closed'];

    public function run(): void
    {
        $users = $this->seedUsers();
        $agents = $users['agents'];
        $allStaff = $users['all'];

        $amenities = $this->seedAmenities();
        $properties = $this->seedProperties($agents, $amenities);
        $clients = $this->seedClients($agents, $allStaff);
        $this->seedLeads($clients, $properties, $agents);
        $this->seedDeals($clients, $properties, $agents, $allStaff);
        $this->seedLeases($properties, $clients);
        $this->seedDocuments($properties, $clients, $allStaff);
        $this->seedAccounts($clients);
    }

    /**
     * @return array{agents: \Illuminate\Support\Collection<int, User>, all: \Illuminate\Support\Collection<int, User>}
     */
    private function seedUsers(): array
    {
        $superAdmin = User::factory()->create([
            'name' => 'Omar Al-Rashid',
            'email' => 'superadmin@softsuitecrm.com',
            'role' => 'super_admin',
            'commission_rate' => 0,
            'joined_at' => '2022-01-15',
        ]);

        $admin = User::factory()->create([
            'name' => 'Fatima Al-Mansoor',
            'email' => 'admin@softsuitecrm.com',
            'role' => 'admin',
            'commission_rate' => 0,
            'joined_at' => '2022-03-01',
        ]);

        $manager = User::factory()->create([
            'name' => 'Khalid Hassan',
            'email' => 'manager@softsuitecrm.com',
            'role' => 'manager',
            'commission_rate' => 2.00,
            'joined_at' => '2022-06-10',
        ]);

        $agent1 = User::factory()->create([
            'name' => 'Sara Ahmed',
            'email' => 'sara@softsuitecrm.com',
            'role' => 'agent',
            'commission_rate' => 3.50,
            'joined_at' => '2023-01-15',
        ]);

        $agent2 = User::factory()->create([
            'name' => 'Youssef Ali',
            'email' => 'youssef@softsuitecrm.com',
            'role' => 'agent',
            'commission_rate' => 3.00,
            'joined_at' => '2023-04-20',
        ]);

        $agent3 = User::factory()->create([
            'name' => 'Layla Ibrahim',
            'email' => 'layla@softsuitecrm.com',
            'role' => 'agent',
            'commission_rate' => 4.00,
            'joined_at' => '2023-09-01',
        ]);

        $agents = collect([$agent1, $agent2, $agent3]);
        $all = collect([$superAdmin, $admin, $manager, $agent1, $agent2, $agent3]);

        return ['agents' => $agents, 'all' => $all];
    }

    /**
     * @return \Illuminate\Support\Collection<int, Amenity>
     */
    private function seedAmenities(): \Illuminate\Support\Collection
    {
        $amenityNames = [
            'Parking', 'Swimming Pool', 'Gym', 'Garden', 'Security',
            'Elevator', 'Balcony', 'Central AC', 'Maid Room', 'Storage Room',
            'Concierge', 'Children Play Area', 'BBQ Area', 'Sauna', 'Jacuzzi',
        ];

        return collect($amenityNames)->map(fn (string $name) => Amenity::create(['name' => $name]));
    }

    /**
     * @param  \Illuminate\Support\Collection<int, User>  $agents
     * @param  \Illuminate\Support\Collection<int, Amenity>  $amenities
     * @return \Illuminate\Support\Collection<int, Property>
     */
    private function seedProperties(\Illuminate\Support\Collection $agents, \Illuminate\Support\Collection $amenities): \Illuminate\Support\Collection
    {
        $propertyData = [
            ['title' => 'Luxury Marina Villa', 'property_type' => 'residential', 'status' => 'available', 'asking_price' => 4500000, 'city' => 'Dubai', 'area' => 'Dubai Marina', 'bedrooms' => 5, 'bathrooms' => 6, 'area_sqft' => 6500, 'year_built' => 2020],
            ['title' => 'Downtown Studio Apartment', 'property_type' => 'residential', 'status' => 'sold', 'asking_price' => 850000, 'city' => 'Dubai', 'area' => 'Downtown Dubai', 'bedrooms' => 0, 'bathrooms' => 1, 'area_sqft' => 550, 'year_built' => 2018],
            ['title' => 'JBR Beachfront 2BR', 'property_type' => 'residential', 'status' => 'available', 'asking_price' => 2200000, 'city' => 'Dubai', 'area' => 'JBR', 'bedrooms' => 2, 'bathrooms' => 2, 'area_sqft' => 1400, 'year_built' => 2016],
            ['title' => 'Business Bay Office Tower', 'property_type' => 'commercial', 'status' => 'available', 'asking_price' => 3200000, 'city' => 'Dubai', 'area' => 'Business Bay', 'bedrooms' => 0, 'bathrooms' => 4, 'area_sqft' => 4200, 'year_built' => 2019],
            ['title' => 'Al Barsha Retail Space', 'property_type' => 'commercial', 'status' => 'rented', 'asking_price' => 1800000, 'city' => 'Dubai', 'area' => 'Al Barsha', 'bedrooms' => 0, 'bathrooms' => 2, 'area_sqft' => 2800, 'year_built' => 2015],
            ['title' => 'Palm Jumeirah Penthouse', 'property_type' => 'residential', 'status' => 'reserved', 'asking_price' => 8500000, 'city' => 'Dubai', 'area' => 'Palm Jumeirah', 'bedrooms' => 4, 'bathrooms' => 5, 'area_sqft' => 5200, 'year_built' => 2022],
            ['title' => 'Silicon Oasis 1BR Rental', 'property_type' => 'rental', 'status' => 'rented', 'asking_price' => 450000, 'city' => 'Dubai', 'area' => 'Dubai Silicon Oasis', 'bedrooms' => 1, 'bathrooms' => 1, 'area_sqft' => 750, 'year_built' => 2017],
            ['title' => 'JLT Office Unit', 'property_type' => 'commercial', 'status' => 'available', 'asking_price' => 950000, 'city' => 'Dubai', 'area' => 'JLT', 'bedrooms' => 0, 'bathrooms' => 2, 'area_sqft' => 1200, 'year_built' => 2014],
            ['title' => 'Sports City 3BR Apartment', 'property_type' => 'rental', 'status' => 'available', 'asking_price' => 750000, 'city' => 'Dubai', 'area' => 'Dubai Sports City', 'bedrooms' => 3, 'bathrooms' => 2, 'area_sqft' => 1800, 'year_built' => 2016],
            ['title' => 'Creek Harbour Waterfront', 'property_type' => 'residential', 'status' => 'available', 'asking_price' => 3800000, 'city' => 'Dubai', 'area' => 'Dubai Creek Harbour', 'bedrooms' => 3, 'bathrooms' => 4, 'area_sqft' => 2800, 'year_built' => 2023],
            ['title' => 'Motor City Townhouse', 'property_type' => 'residential', 'status' => 'sold', 'asking_price' => 1650000, 'city' => 'Dubai', 'area' => 'Motor City', 'bedrooms' => 3, 'bathrooms' => 3, 'area_sqft' => 2200, 'year_built' => 2012],
            ['title' => 'DIFC Commercial Suite', 'property_type' => 'commercial', 'status' => 'available', 'asking_price' => 5200000, 'city' => 'Dubai', 'area' => 'DIFC', 'bedrooms' => 0, 'bathrooms' => 3, 'area_sqft' => 3500, 'year_built' => 2021],
            ['title' => 'Arabian Ranches Villa', 'property_type' => 'residential', 'status' => 'available', 'asking_price' => 2900000, 'city' => 'Dubai', 'area' => 'Arabian Ranches', 'bedrooms' => 4, 'bathrooms' => 4, 'area_sqft' => 3800, 'year_built' => 2010],
            ['title' => 'Discovery Gardens 2BR', 'property_type' => 'rental', 'status' => 'rented', 'asking_price' => 380000, 'city' => 'Dubai', 'area' => 'Discovery Gardens', 'bedrooms' => 2, 'bathrooms' => 2, 'area_sqft' => 1100, 'year_built' => 2009],
            ['title' => 'Deira Gold Souk Shop', 'property_type' => 'commercial', 'status' => 'rented', 'asking_price' => 1200000, 'city' => 'Dubai', 'area' => 'Deira', 'bedrooms' => 0, 'bathrooms' => 1, 'area_sqft' => 800, 'year_built' => 2005],
            ['title' => 'Al Furjan 4BR Villa', 'property_type' => 'residential', 'status' => 'available', 'asking_price' => 2100000, 'city' => 'Dubai', 'area' => 'Al Furjan', 'bedrooms' => 4, 'bathrooms' => 5, 'area_sqft' => 3200, 'year_built' => 2019],
            ['title' => 'Mirdif Community 3BR', 'property_type' => 'rental', 'status' => 'available', 'asking_price' => 620000, 'city' => 'Dubai', 'area' => 'Mirdif', 'bedrooms' => 3, 'bathrooms' => 3, 'area_sqft' => 2000, 'year_built' => 2013],
            ['title' => 'Bluewaters Island 2BR', 'property_type' => 'residential', 'status' => 'reserved', 'asking_price' => 3100000, 'city' => 'Dubai', 'area' => 'Bluewaters Island', 'bedrooms' => 2, 'bathrooms' => 3, 'area_sqft' => 1600, 'year_built' => 2021],
            ['title' => 'International City Studio', 'property_type' => 'rental', 'status' => 'rented', 'asking_price' => 220000, 'city' => 'Dubai', 'area' => 'International City', 'bedrooms' => 0, 'bathrooms' => 1, 'area_sqft' => 400, 'year_built' => 2008],
            ['title' => 'Jumeirah Village Circle 1BR', 'property_type' => 'residential', 'status' => 'sold', 'asking_price' => 680000, 'city' => 'Dubai', 'area' => 'JVC', 'bedrooms' => 1, 'bathrooms' => 1, 'area_sqft' => 850, 'year_built' => 2020],
        ];

        $properties = collect();

        foreach ($propertyData as $index => $data) {
            $property = Property::factory()->create([
                ...$data,
                'address' => fake()->streetAddress().', '.$data['area'].', '.$data['city'],
                'description' => fake()->paragraphs(2, true),
                'latitude' => fake()->latitude(25.0, 25.3),
                'longitude' => fake()->longitude(55.1, 55.4),
                'agent_id' => $agents[$index % 3]->id,
            ]);

            // Primary photo
            PropertyPhoto::factory()->primary()->create(['property_id' => $property->id]);

            // 2-4 additional photos
            PropertyPhoto::factory()
                ->count(fake()->numberBetween(2, 4))
                ->create(['property_id' => $property->id]);

            // Attach 3-6 random amenities
            $property->amenities()->attach(
                $amenities->random(fake()->numberBetween(3, 6))->pluck('id')
            );

            $properties->push($property);
        }

        return $properties;
    }

    /**
     * @param  \Illuminate\Support\Collection<int, User>  $agents
     * @param  \Illuminate\Support\Collection<int, User>  $allStaff
     * @return \Illuminate\Support\Collection<int, Client>
     */
    private function seedClients(\Illuminate\Support\Collection $agents, \Illuminate\Support\Collection $allStaff): \Illuminate\Support\Collection
    {
        $clientTypes = ['buyer', 'buyer', 'buyer', 'seller', 'seller', 'tenant', 'tenant', 'tenant', 'landlord', 'landlord'];
        $clients = collect();

        for ($i = 0; $i < 30; $i++) {
            $client = Client::factory()->create([
                'client_type' => $clientTypes[$i % count($clientTypes)],
                'agent_id' => $agents[$i % 3]->id,
            ]);

            // 1-4 interactions per client
            $interactionCount = fake()->numberBetween(1, 4);

            for ($j = 0; $j < $interactionCount; $j++) {
                ClientInteraction::factory()->create([
                    'client_id' => $client->id,
                    'recorded_by' => $allStaff->random()->id,
                    'created_at' => fake()->dateTimeBetween('-6 months', 'now'),
                ]);
            }

            $clients->push($client);
        }

        return $clients;
    }

    /**
     * @param  \Illuminate\Support\Collection<int, Client>  $clients
     * @param  \Illuminate\Support\Collection<int, Property>  $properties
     * @param  \Illuminate\Support\Collection<int, User>  $agents
     */
    private function seedLeads(
        \Illuminate\Support\Collection $clients,
        \Illuminate\Support\Collection $properties,
        \Illuminate\Support\Collection $agents,
    ): void {
        $statuses = ['new', 'new', 'new', 'new', 'new', 'contacted', 'contacted', 'contacted', 'contacted', 'contacted', 'qualified', 'qualified', 'qualified', 'qualified', 'qualified', 'lost', 'lost', 'lost', 'lost', 'lost', 'converted', 'converted', 'converted', 'converted', 'converted'];
        $sources = ['website', 'referral', 'walk_in', 'social_media', 'portal', 'other'];

        for ($i = 0; $i < 25; $i++) {
            $budgetMin = fake()->randomFloat(2, 200000, 2000000);

            Lead::factory()->create([
                'client_id' => fake()->boolean(80) ? $clients->random()->id : null,
                'property_id' => fake()->boolean(70) ? $properties->random()->id : null,
                'source' => $sources[$i % count($sources)],
                'status' => $statuses[$i],
                'budget_min' => $budgetMin,
                'budget_max' => $budgetMin + fake()->randomFloat(2, 100000, 1500000),
                'next_follow_up_at' => in_array($statuses[$i], ['new', 'contacted', 'qualified'])
                    ? fake()->dateTimeBetween('now', '+14 days')
                    : null,
                'agent_id' => $agents[$i % 3]->id,
                'notes' => fake()->optional(0.6)->paragraph(),
            ]);
        }
    }

    /**
     * @param  \Illuminate\Support\Collection<int, Client>  $clients
     * @param  \Illuminate\Support\Collection<int, Property>  $properties
     * @param  \Illuminate\Support\Collection<int, User>  $agents
     * @param  \Illuminate\Support\Collection<int, User>  $allStaff
     */
    private function seedDeals(
        \Illuminate\Support\Collection $clients,
        \Illuminate\Support\Collection $properties,
        \Illuminate\Support\Collection $agents,
        \Illuminate\Support\Collection $allStaff,
    ): void {
        // Distribute 15 deals across all 7 stages: 2-3 per stage
        $stageDistribution = ['inquiry', 'inquiry', 'viewing', 'viewing', 'offer', 'offer', 'negotiation', 'negotiation', 'agreement', 'agreement', 'closing', 'closing', 'closed', 'closed', 'closed'];

        $buyerClients = $clients->filter(fn (Client $c) => in_array($c->client_type, ['buyer', 'tenant']));

        for ($i = 0; $i < 15; $i++) {
            $property = $properties->random();
            $client = $buyerClients->random();
            $agent = $agents[$i % 3];
            $stage = $stageDistribution[$i];
            $dealValue = (float) $property->asking_price * fake()->randomFloat(2, 0.85, 1.05);
            $commissionRate = (float) $agent->commission_rate / 100;

            $deal = Deal::factory()->create([
                'title' => $client->name.' — '.$property->title,
                'property_id' => $property->id,
                'client_id' => $client->id,
                'agent_id' => $agent->id,
                'stage' => $stage,
                'deal_value' => round($dealValue, 2),
                'commission_amount' => round($dealValue * $commissionRate, 2),
                'expected_closing_date' => fake()->dateTimeBetween('now', '+6 months'),
                'notes' => fake()->optional(0.5)->paragraph(),
            ]);

            // Build stage history up to current stage
            $currentStageIndex = array_search($stage, $this->dealStages);
            $historyDate = Carbon::now()->subDays(($currentStageIndex + 1) * fake()->numberBetween(3, 10));

            for ($j = 0; $j <= $currentStageIndex; $j++) {
                DealStageHistory::create([
                    'deal_id' => $deal->id,
                    'from_stage' => $j === 0 ? null : $this->dealStages[$j - 1],
                    'to_stage' => $this->dealStages[$j],
                    'changed_by' => $allStaff->random()->id,
                    'note' => fake()->optional(0.4)->sentence(),
                    'created_at' => $historyDate->copy()->addDays($j * fake()->numberBetween(2, 7)),
                    'updated_at' => $historyDate->copy()->addDays($j * fake()->numberBetween(2, 7)),
                ]);
            }
        }
    }

    /**
     * @param  \Illuminate\Support\Collection<int, Property>  $properties
     * @param  \Illuminate\Support\Collection<int, Client>  $clients
     */
    private function seedLeases(
        \Illuminate\Support\Collection $properties,
        \Illuminate\Support\Collection $clients,
    ): void {
        $rentalProperties = $properties->filter(fn (Property $p) => in_array($p->status, ['rented', 'available']) && in_array($p->property_type, ['rental', 'commercial']));
        $tenantClients = $clients->filter(fn (Client $c) => $c->client_type === 'tenant');

        // Pad with buyer clients if not enough tenants
        if ($tenantClients->count() < 10) {
            $extraClients = $clients->filter(fn (Client $c) => $c->client_type === 'buyer')->take(10 - $tenantClients->count());
            $tenantClients = $tenantClients->merge($extraClients);
        }

        for ($i = 0; $i < 10; $i++) {
            $property = $rentalProperties->count() > $i
                ? $rentalProperties->values()->get($i)
                : $properties->random();

            $tenant = $tenantClients->values()->get($i % $tenantClients->count());
            $startDate = Carbon::now()->subMonths(fake()->numberBetween(1, 10));
            $endDate = $startDate->copy()->addMonths(12);
            $monthlyRent = fake()->randomElement([3000, 4500, 5000, 6500, 8000, 10000, 12000, 15000, 18000, 22000]);

            $lease = Lease::factory()->create([
                'property_id' => $property->id,
                'client_id' => $tenant->id,
                'start_date' => $startDate,
                'end_date' => $endDate,
                'monthly_rent' => $monthlyRent,
                'security_deposit' => $monthlyRent * 2,
                'status' => 'active',
                'is_renewable' => fake()->boolean(40),
            ]);

            // Generate full payment schedule from start to end date
            $paymentDate = $startDate->copy();
            $now = Carbon::now();

            while ($paymentDate->lt($endDate)) {
                $isPast = $paymentDate->lt($now);

                LeasePayment::create([
                    'lease_id' => $lease->id,
                    'due_date' => $paymentDate->copy(),
                    'amount_due' => $monthlyRent,
                    'is_paid' => $isPast ? fake()->boolean(90) : false,
                    'paid_at' => $isPast && fake()->boolean(90)
                        ? $paymentDate->copy()->addDays(fake()->numberBetween(0, 5))
                        : null,
                ]);

                $paymentDate->addMonth();
            }
        }
    }

    /**
     * @param  \Illuminate\Support\Collection<int, Property>  $properties
     * @param  \Illuminate\Support\Collection<int, Client>  $clients
     * @param  \Illuminate\Support\Collection<int, User>  $allStaff
     */
    private function seedDocuments(
        \Illuminate\Support\Collection $properties,
        \Illuminate\Support\Collection $clients,
        \Illuminate\Support\Collection $allStaff,
    ): void {
        // Property documents
        $properties->take(10)->each(function (Property $property) use ($allStaff) {
            Document::factory()->create([
                'name' => 'Title Deed — '.$property->title,
                'type' => 'title_deed',
                'documentable_type' => Property::class,
                'documentable_id' => $property->id,
                'uploaded_by' => $allStaff->random()->id,
            ]);
        });

        // Client ID documents
        $clients->take(15)->each(function (Client $client) use ($allStaff) {
            Document::factory()->create([
                'name' => 'National ID — '.$client->name,
                'type' => 'client_id_doc',
                'documentable_type' => Client::class,
                'documentable_id' => $client->id,
                'uploaded_by' => $allStaff->random()->id,
            ]);
        });
    }

    /**
     * @param  \Illuminate\Support\Collection<int, Client>  $clients
     */
    private function seedAccounts(\Illuminate\Support\Collection $clients): void
    {
        // System accounts
        $systemAccounts = [
            ['name' => 'Commission Income', 'code' => 'SYS-001', 'account_type' => 'income', 'account_sub_type' => 'commission', 'balance' => 245000],
            ['name' => 'Rental Income', 'code' => 'SYS-002', 'account_type' => 'income', 'account_sub_type' => 'rental_income', 'balance' => 890000],
            ['name' => 'Security Deposits Held', 'code' => 'SYS-003', 'account_type' => 'liability', 'account_sub_type' => 'security_deposit', 'balance' => 180000],
            ['name' => 'Maintenance Reserve', 'code' => 'SYS-004', 'account_type' => 'expense', 'account_sub_type' => 'maintenance', 'balance' => 35000],
            ['name' => 'Accounts Receivable', 'code' => 'SYS-005', 'account_type' => 'asset', 'account_sub_type' => 'receivable', 'balance' => 420000],
        ];

        foreach ($systemAccounts as $accData) {
            $account = Account::factory()->systemCreated()->create($accData);

            // Add 3-5 transactions per system account
            $transactionCount = fake()->numberBetween(3, 5);
            for ($i = 0; $i < $transactionCount; $i++) {
                AccountTransaction::factory()->create([
                    'account_id' => $account->id,
                    'type' => fake()->randomElement(['debit', 'credit']),
                    'amount' => fake()->randomFloat(2, 1000, 50000),
                    'description' => fake()->sentence(),
                    'transaction_date' => fake()->dateTimeBetween('-6 months', 'now'),
                ]);
            }
        }

        // Client-linked accounts for first 10 clients
        $clients->take(10)->each(function (Client $client) {
            $account = Account::factory()->create([
                'name' => $client->name.' — Receivable',
                'code' => 'CLI-'.str_pad((string) $client->id, 4, '0', STR_PAD_LEFT),
                'account_type' => 'asset',
                'account_sub_type' => 'receivable',
                'balance' => fake()->randomFloat(2, 0, 100000),
                'is_system_created' => false,
                'client_id' => $client->id,
            ]);

            AccountTransaction::factory()->count(fake()->numberBetween(1, 3))->create([
                'account_id' => $account->id,
                'transaction_date' => fake()->dateTimeBetween('-6 months', 'now'),
            ]);
        });
    }
}
